<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Project;
use App\Models\Category;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;

class SubmitProject extends Component
{
    public $name;
    public $description;
    public $website_url;
    public $category_id;
    public $privacy_level;
    public $registration_required = false;
    public $no_log_policy = false;
    public $source_code_availability = false;
    public $own_liquidity = false;
    
    // Arrays for multi-selects (simplified for now as comma strings or JSON)
    public $supported_cryptos = []; 
    public $features = [];

    protected $rules = [
        'name' => 'required|string|max:255|unique:projects,name',
        'description' => 'required|string|min:50',
        'website_url' => 'required|url',
        'category_id' => 'required|exists:categories,id',
        'privacy_level' => 'nullable|string',
        'supported_cryptos' => 'nullable', // Handle as array logic later
    ];

    public function submit()
    {
        $this->validate();

        $slug = Str::slug($this->name);

        Project::create([
            'user_id' => Auth::id(),
            'name' => $this->name,
            'slug' => $slug,
            'description' => $this->description,
            'website_url' => $this->website_url,
            'category_id' => $this->category_id,
            'list_status' => 'proposed', // Default
            'online_status' => 'offline', // Checked by bot later
            'privacy_level' => $this->privacy_level,
            'registration_required' => $this->registration_required,
            'no_log_policy' => $this->no_log_policy,
            'source_code_availability' => $this->source_code_availability,
            'own_liquidity' => $this->own_liquidity,
            // Defaults
            'star_rating' => 0,
            'review_count' => 0,
            'trust_score' => 50, // Neutral start
        ]);

        session()->flash('message', 'Project submitted successfully! It is now pending approval.');
        
        return redirect()->route('home');
    }

    public function render()
    {
        return view('livewire.submit-project', [
            'categories' => Category::orderBy('name')->get()
        ]);
    }
}
