<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Project;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class MonitorProjectHealth extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'monitor:health';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check online status of all listed projects';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $projects = Project::whereNotNull('website_url')->get();
        $this->info("Starting health check for {$projects->count()} projects...");

        foreach ($projects as $project) {
            $this->checkProject($project);
        }

        $this->info("Health check completed.");
    }

    private function checkProject($project)
    {
        try {
            // Set a timeout to avoid hanging on dead sites
            $response = Http::timeout(10)->get($project->website_url);

            if ($response->successful()) {
                $status = 'online';
                $this->line("✅ {$project->name}: Online ({$response->status()})");
            } else {
                $status = 'offline';
                $this->error("❌ {$project->name}: Offline ({$response->status()})");
            }
        } catch (\Exception $e) {
            $status = 'offline';
            $this->error("❌ {$project->name}: Error ({$e->getMessage()})");
        }

        // Only update if status changed to avoid unnecessary DB writes
        if ($project->online_status !== $status) {
            $project->update(['online_status' => $status]);
            Log::info("Project {$project->name} status changed to {$status}");
        }
    }
}
